<?php

/**
 * The core dual currency functionality.
 *
 * @link       https://webtospec.com
 * @since      1.0.0
 *
 * @package    Woo_Bg_Dual_Currency
 * @subpackage Woo_Bg_Dual_Currency/includes
 */

/**
 * The core dual currency class.
 *
 * This class handles all the dual currency conversion and formatting logic.
 *
 * @since      1.0.0
 * @package    Woo_Bg_Dual_Currency
 * @subpackage Woo_Bg_Dual_Currency/includes
 * @author     Web To Spec <grow@webtospec.com>
 */
class Woo_Bg_Dual_Currency_Core
{

    /**
     * The conversion rate from BGN to EUR.
     * Official rate: 1 BGN = 0.511292 EUR (1 EUR = 1.956 BGN)
     *
     * @since    1.0.0
     * @access   private
     * @var      float    $conversion_rate    The BGN to EUR conversion rate.
     */
    private static $conversion_rate = null;

    /**
     * Get the conversion rate from settings or default.
     *
     * @since    1.0.0
     * @return   float    The conversion rate.
     */
    private static function get_rate()
    {
        if (self::$conversion_rate === null) {
            self::$conversion_rate = get_option('woo_bg_dual_currency_conversion_rate', 0.511292);
        }
        return self::$conversion_rate;
    }

    /**
     * Check if the current store currency is Bulgarian Lev.
     *
     * @since    1.0.0
     * @return   bool    True if store currency is BGN, false otherwise.
     */
    public static function is_bgn_currency()
    {
        return get_woocommerce_currency() === 'BGN';
    }

    /**
     * Convert BGN amount to EUR.
     *
     * @since    1.0.0
     * @param    float    $bgn_amount    The amount in BGN.
     * @return   float    The converted amount in EUR.
     */
    public static function convert_bgn_to_eur($bgn_amount)
    {
        return round($bgn_amount * self::get_rate(), 2);
    }

    /**
     * Format dual currency display.
     *
     * @since    1.0.0
     * @param    string   $price_html    The original price HTML.
     * @param    float    $price         The price amount.
     * @return   string   The formatted dual currency HTML.
     */
    public static function format_dual_currency($price_html, $price = null)
    {
        // Only proceed if store currency is BGN
        if (! self::is_bgn_currency()) {
            return $price_html;
        }

        // If price is not provided, try to extract it from the HTML
        if ($price === null) {
            $price = self::extract_price_from_html($price_html);
        }

        // If we still don't have a price, return original
        if ($price === null || $price <= 0) {
            return $price_html;
        }

        // Convert to EUR
        $eur_price = self::convert_bgn_to_eur($price);

        // Create dual currency display
        $bgn_formatted = wc_price($price, array('currency' => 'BGN'));
        $eur_formatted = number_format($eur_price, 2, '.', '');

        // Replace the original price with dual currency format
        $dual_currency = $bgn_formatted . ' / € ' . $eur_formatted;

        return $dual_currency;
    }

    /**
     * Extract numeric price from formatted price HTML.
     *
     * @since    1.0.0
     * @param    string   $price_html    The price HTML string.
     * @return   float|null    The extracted price or null if not found.
     */
    public static function extract_price_from_html($price_html)
    {
        // Remove HTML tags and get clean text
        $clean_text = wp_strip_all_tags($price_html);

        // Extract numeric value using regex
        preg_match('/[\d,]+\.?\d*/', $clean_text, $matches);

        if (! empty($matches[0])) {
            // Remove commas and convert to float
            $price = str_replace(',', '', $matches[0]);
            return floatval($price);
        }

        return null;
    }

    /**
     * Format price range for dual currency.
     *
     * @since    1.0.0
     * @param    float    $min_price    The minimum price.
     * @param    float    $max_price    The maximum price.
     * @return   string   The formatted dual currency price range.
     */
    public static function format_price_range($min_price, $max_price)
    {
        if (! self::is_bgn_currency()) {
            return wc_format_price_range($min_price, $max_price);
        }

        $min_eur = self::convert_bgn_to_eur($min_price);
        $max_eur = self::convert_bgn_to_eur($max_price);

        $bgn_range = wc_format_price_range($min_price, $max_price);
        $eur_range = '€ ' . number_format($min_eur, 2) . ' - € ' . number_format($max_eur, 2);

        return $bgn_range . ' / ' . $eur_range;
    }

    /**
     * Get dual currency symbol.
     *
     * @since    1.0.0
     * @return   string    The dual currency symbol.
     */
    public static function get_dual_currency_symbol()
    {
        if (! self::is_bgn_currency()) {
            return get_woocommerce_currency_symbol();
        }

        return 'лв. / €';
    }

    /**
     * Check if dual currency should be displayed in emails.
     *
     * @since    1.0.0
     * @return   bool    True if dual currency should be shown in emails.
     */
    public static function show_in_emails()
    {
        return self::is_bgn_currency() && get_option('woo_bg_dual_currency_show_in_emails', 1);
    }

    /**
     * Check if dual currency should be displayed in admin.
     *
     * @since    1.0.0
     * @return   bool    True if dual currency should be shown in admin.
     */
    public static function show_in_admin()
    {
        return self::is_bgn_currency() && get_option('woo_bg_dual_currency_show_in_admin', 1);
    }

    /**
     * Get the conversion rate.
     *
     * @since    1.0.0
     * @return   float    The current conversion rate.
     */
    public static function get_conversion_rate()
    {
        return self::get_rate();
    }

    /**
     * Set a custom conversion rate.
     *
     * @since    1.0.0
     * @param    float    $rate    The new conversion rate.
     */
    public static function set_conversion_rate($rate)
    {
        if (is_numeric($rate) && $rate > 0) {
            self::$conversion_rate = floatval($rate);
            update_option('woo_bg_dual_currency_conversion_rate', self::$conversion_rate);
        }
    }
}
