<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://webtospec.com
 * @since      1.0.0
 *
 * @package    Woo_Bg_Dual_Currency
 * @subpackage Woo_Bg_Dual_Currency/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Woo_Bg_Dual_Currency
 * @subpackage Woo_Bg_Dual_Currency/admin
 * @author     Web To Spec <grow@webtospec.com>
 */
class Woo_Bg_Dual_Currency_Admin
{

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct($plugin_name, $version)
	{

		$this->plugin_name = $plugin_name;
		$this->version = $version;
	}

	/**
	 * Display admin notices.
	 *
	 * @since    1.0.0
	 */
	public function admin_notices()
	{
		// Check for activation notice
		if (get_transient('woo_bg_dual_currency_activation_notice')) {
?>
			<div class="notice notice-warning is-dismissible">
				<p><strong><?php _e('Bulgaria Dual Currency', 'woo-bg-dual-currency'); ?>:</strong>
					<?php _e('This plugin requires WooCommerce to be installed and activated.', 'woo-bg-dual-currency'); ?>
				</p>
			</div>
		<?php
			delete_transient('woo_bg_dual_currency_activation_notice');
		}

		// Check if current page is plugin settings page
		$screen = get_current_screen();
		if (isset($screen->id) && strpos($screen->id, 'woo-bg-dual-currency') !== false) {
			// Test if the plugin is working
		?>
			<div class="notice notice-info">
				<p><strong><?php _e('Plugin Status:', 'woo-bg-dual-currency'); ?></strong>
					<?php if (class_exists('WooCommerce')): ?>
						<?php _e('WooCommerce detected ✓', 'woo-bg-dual-currency'); ?>
					<?php else: ?>
						<?php _e('WooCommerce not found ✗', 'woo-bg-dual-currency'); ?>
					<?php endif; ?>
				</p>
			</div>
<?php
		}
	}

	/**
	 * Initialize admin hooks for dual currency display.
	 *
	 * @since    1.0.0
	 */
	public function init_admin_hooks()
	{

		// Only add hooks if WooCommerce is active
		if (! class_exists('WooCommerce')) {
			return;
		}

		// Admin order items - add EUR display to line items cost
		add_action('woocommerce_before_order_itemmeta', array($this, 'add_eur_line_item_cost'), 10, 3);

		// Admin order items - add EUR display to order totals
		add_action('woocommerce_admin_order_totals_after_total', array($this, 'add_eur_order_total'));
		add_action('woocommerce_admin_order_totals_after_subtotal', array($this, 'add_eur_subtotal'));
		add_action('woocommerce_admin_order_totals_after_tax', array($this, 'add_eur_tax_total'));
		add_action('woocommerce_admin_order_totals_after_shipping', array($this, 'add_eur_shipping_total'));

		// Admin product edit page
		add_action('woocommerce_product_options_pricing', array($this, 'add_product_eur_price_display'));
	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles()
	{

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Woo_Bg_Dual_Currency_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Woo_Bg_Dual_Currency_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/woo-bg-dual-currency-admin.css', array(), $this->version, 'all');
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts()
	{

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Woo_Bg_Dual_Currency_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Woo_Bg_Dual_Currency_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/woo-bg-dual-currency-admin.js', array('jquery'), $this->version, false);
	}

	/**
	 * Add admin menu for plugin settings.
	 *
	 * @since    1.0.0
	 */
	public function add_admin_menu()
	{
		// Debug: Check if WooCommerce is active
		if (! class_exists('WooCommerce')) {
			// If WooCommerce is not active, add to main admin menu instead
			add_options_page(
				__('Dual Currency Settings', 'woo-bg-dual-currency'),
				__('Dual Currency', 'woo-bg-dual-currency'),
				'manage_options',
				'woo-bg-dual-currency',
				array($this, 'admin_page_callback')
			);
		} else {
			// Add to WooCommerce menu
			add_submenu_page(
				'woocommerce',
				__('Dual Currency Settings', 'woo-bg-dual-currency'),
				__('Dual Currency', 'woo-bg-dual-currency'),
				'manage_woocommerce',
				'woo-bg-dual-currency',
				array($this, 'admin_page_callback')
			);
		}
	}

	/**
	 * Initialize admin settings.
	 *
	 * @since    1.0.0
	 */
	public function init_admin_settings()
	{
		register_setting('woo_bg_dual_currency_settings', 'woo_bg_dual_currency_conversion_rate');
		register_setting('woo_bg_dual_currency_settings', 'woo_bg_dual_currency_show_in_emails');
		register_setting('woo_bg_dual_currency_settings', 'woo_bg_dual_currency_show_in_admin');

		add_settings_section(
			'woo_bg_dual_currency_main_section',
			__('Dual Currency Settings', 'woo-bg-dual-currency'),
			array($this, 'settings_section_callback'),
			'woo_bg_dual_currency_settings'
		);

		add_settings_field(
			'conversion_rate',
			__('BGN to EUR Conversion Rate', 'woo-bg-dual-currency'),
			array($this, 'conversion_rate_callback'),
			'woo_bg_dual_currency_settings',
			'woo_bg_dual_currency_main_section'
		);

		add_settings_field(
			'show_in_emails',
			__('Show in Emails', 'woo-bg-dual-currency'),
			array($this, 'show_in_emails_callback'),
			'woo_bg_dual_currency_settings',
			'woo_bg_dual_currency_main_section'
		);

		add_settings_field(
			'show_in_admin',
			__('Show in Admin', 'woo-bg-dual-currency'),
			array($this, 'show_in_admin_callback'),
			'woo_bg_dual_currency_settings',
			'woo_bg_dual_currency_main_section'
		);
	}

	/**
	 * Admin page callback.
	 *
	 * @since    1.0.0
	 */
	public function admin_page_callback()
	{
		include_once plugin_dir_path(__FILE__) . 'partials/woo-bg-dual-currency-admin-display.php';
	}

	/**
	 * Settings section callback.
	 *
	 * @since    1.0.0
	 */
	public function settings_section_callback()
	{
		echo '<p>' . __('Configure the dual currency display settings for your WooCommerce store.', 'woo-bg-dual-currency') . '</p>';
	}

	/**
	 * Conversion rate field callback.
	 *
	 * @since    1.0.0
	 */
	public function conversion_rate_callback()
	{
		$rate = get_option('woo_bg_dual_currency_conversion_rate', 0.511292);
		echo '<input type="number" step="0.000001" name="woo_bg_dual_currency_conversion_rate" value="' . esc_attr($rate) . '" />';
		echo '<p class="description">' . __('Official rate: 1 BGN = 0.511292 EUR (1 EUR = 1.956 BGN)', 'woo-bg-dual-currency') . '</p>';
	}

	/**
	 * Show in emails field callback.
	 *
	 * @since    1.0.0
	 */
	public function show_in_emails_callback()
	{
		$show = get_option('woo_bg_dual_currency_show_in_emails', 1);
		echo '<input type="checkbox" name="woo_bg_dual_currency_show_in_emails" value="1" ' . checked(1, $show, false) . ' />';
		echo '<label>' . __('Display dual currency in order confirmation emails', 'woo-bg-dual-currency') . '</label>';
	}

	/**
	 * Show in admin field callback.
	 *
	 * @since    1.0.0
	 */
	public function show_in_admin_callback()
	{
		$show = get_option('woo_bg_dual_currency_show_in_admin', 1);
		echo '<input type="checkbox" name="woo_bg_dual_currency_show_in_admin" value="1" ' . checked(1, $show, false) . ' />';
		echo '<label>' . __('Display dual currency in admin order pages', 'woo-bg-dual-currency') . '</label>';
	}

	/**
	 * Add EUR equivalent to line item cost display.
	 *
	 * @since    1.0.0
	 * @param    int        $item_id   Item ID.
	 * @param    array      $item      Order item.
	 * @param    WC_Product $product   Product object.
	 */
	public function add_eur_line_item_cost($item_id, $item, $product)
	{
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency() || ! Woo_Bg_Dual_Currency_Core::show_in_admin()) {
			return;
		}

		if (is_object($item) && method_exists($item, 'get_total')) {
			$bgn_total = $item->get_total();
			$eur_total = Woo_Bg_Dual_Currency_Core::convert_bgn_to_eur($bgn_total);
			echo '<div class="woo-bg-dual-currency-line-item" style="font-size: 11px; color: #666; margin-top: 2px;">';
			echo __('EUR:', 'woo-bg-dual-currency') . ' € ' . number_format($eur_total, 2);
			echo '</div>';
		}
	}

	/**
	 * Add EUR equivalent to order total.
	 *
	 * @since    1.0.0
	 * @param    int $order_id Order ID.
	 */
	public function add_eur_order_total($order_id)
	{
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency() || ! Woo_Bg_Dual_Currency_Core::show_in_admin()) {
			return;
		}

		$order = wc_get_order($order_id);
		if (! $order) {
			return;
		}

		$bgn_total = $order->get_total();
		$eur_total = Woo_Bg_Dual_Currency_Core::convert_bgn_to_eur($bgn_total);

		echo '<tr>';
		echo '<td class="label">' . __('Total (EUR):', 'woo-bg-dual-currency') . '</td>';
		echo '<td width="1%"></td>';
		echo '<td class="total">€ ' . number_format($eur_total, 2) . '</td>';
		echo '</tr>';
	}

	/**
	 * Add EUR equivalent to order subtotal.
	 *
	 * @since    1.0.0
	 * @param    int $order_id Order ID.
	 */
	public function add_eur_subtotal($order_id)
	{
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency() || ! Woo_Bg_Dual_Currency_Core::show_in_admin()) {
			return;
		}

		$order = wc_get_order($order_id);
		if (! $order) {
			return;
		}

		$bgn_subtotal = $order->get_subtotal();
		$eur_subtotal = Woo_Bg_Dual_Currency_Core::convert_bgn_to_eur($bgn_subtotal);

		echo '<tr>';
		echo '<td class="label">' . __('Subtotal (EUR):', 'woo-bg-dual-currency') . '</td>';
		echo '<td width="1%"></td>';
		echo '<td class="total">€ ' . number_format($eur_subtotal, 2) . '</td>';
		echo '</tr>';
	}

	/**
	 * Add EUR equivalent to tax total.
	 *
	 * @since    1.0.0
	 * @param    int $order_id Order ID.
	 */
	public function add_eur_tax_total($order_id)
	{
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency() || ! Woo_Bg_Dual_Currency_Core::show_in_admin()) {
			return;
		}

		$order = wc_get_order($order_id);
		if (! $order || $order->get_total_tax() <= 0) {
			return;
		}

		$bgn_tax = $order->get_total_tax();
		$eur_tax = Woo_Bg_Dual_Currency_Core::convert_bgn_to_eur($bgn_tax);

		echo '<tr>';
		echo '<td class="label">' . __('Tax (EUR):', 'woo-bg-dual-currency') . '</td>';
		echo '<td width="1%"></td>';
		echo '<td class="total">€ ' . number_format($eur_tax, 2) . '</td>';
		echo '</tr>';
	}

	/**
	 * Add EUR equivalent to shipping total.
	 *
	 * @since    1.0.0
	 * @param    int $order_id Order ID.
	 */
	public function add_eur_shipping_total($order_id)
	{
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency() || ! Woo_Bg_Dual_Currency_Core::show_in_admin()) {
			return;
		}

		$order = wc_get_order($order_id);
		if (! $order || $order->get_shipping_total() <= 0) {
			return;
		}

		$bgn_shipping = $order->get_shipping_total();
		$eur_shipping = Woo_Bg_Dual_Currency_Core::convert_bgn_to_eur($bgn_shipping);

		echo '<tr>';
		echo '<td class="label">' . __('Shipping (EUR):', 'woo-bg-dual-currency') . '</td>';
		echo '<td width="1%"></td>';
		echo '<td class="total">€ ' . number_format($eur_shipping, 2) . '</td>';
		echo '</tr>';
	}

	/**
	 * Add EUR price display in product edit page.
	 *
	 * @since    1.0.0
	 */
	public function add_product_eur_price_display()
	{
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency()) {
			return;
		}

		global $product_object;

		if (! $product_object) {
			return;
		}

		$price = $product_object->get_price();

		if ($price) {
			$eur_price = Woo_Bg_Dual_Currency_Core::convert_bgn_to_eur($price);
			echo '<p class="form-field"><strong>' . __('EUR Equivalent:', 'woo-bg-dual-currency') . '</strong> € ' . number_format($eur_price, 2) . '</p>';
		}
	}
}
