<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://webtospec.com
 * @since      1.0.0
 *
 * @package    Woo_Bg_Dual_Currency
 * @subpackage Woo_Bg_Dual_Currency/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Woo_Bg_Dual_Currency
 * @subpackage Woo_Bg_Dual_Currency/public
 * @author     Web To Spec <grow@webtospec.com>
 */
class Woo_Bg_Dual_Currency_Public
{

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct($plugin_name, $version)
	{

		$this->plugin_name = $plugin_name;
		$this->version = $version;
	}

	/**
	 * Initialize WooCommerce hooks for dual currency display.
	 *
	 * @since    1.0.0
	 */
	public function init_woocommerce_hooks()
	{

		// Only add hooks if WooCommerce is active
		if (! class_exists('WooCommerce')) {
			return;
		}

		// Main price display filter
		add_filter('woocommerce_get_price_html', array($this, 'display_dual_currency_price'), 10, 2);

		// Add cart hooks now that main price filter works
		add_filter('woocommerce_cart_item_price', array($this, 'display_dual_currency_cart_price'), 10, 3);
		add_filter('woocommerce_cart_item_subtotal', array($this, 'display_dual_currency_cart_subtotal'), 10, 3);

		// Add variable and grouped product support
		add_filter('woocommerce_variable_price_html', array($this, 'display_dual_currency_variable_price'), 10, 2);
		add_filter('woocommerce_grouped_price_html', array($this, 'display_dual_currency_grouped_price'), 10, 2);

		// Cart and Checkout totals
		add_filter('woocommerce_cart_subtotal', array($this, 'display_dual_currency_cart_subtotal_total'), 10, 3);
		add_filter('woocommerce_cart_shipping_method_full_label', array($this, 'display_dual_currency_shipping'), 10, 2);
		add_filter('woocommerce_cart_totals_order_total_html', array($this, 'display_dual_currency_cart_order_total'), 10, 1);

		// Add EUR conversion to order received page via filters
		add_filter('woocommerce_order_formatted_line_subtotal', array($this, 'add_eur_to_line_subtotal'), 10, 3);
	}
	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles()
	{

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Woo_Bg_Dual_Currency_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Woo_Bg_Dual_Currency_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/woo-bg-dual-currency-public.css', array(), $this->version, 'all');
	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts()
	{

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Woo_Bg_Dual_Currency_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Woo_Bg_Dual_Currency_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/woo-bg-dual-currency-public.js', array('jquery'), $this->version, false);

		// Localize script with conversion rate data
		if (Woo_Bg_Dual_Currency_Core::is_bgn_currency()) {
			wp_localize_script($this->plugin_name, 'wooBgDualCurrency', array(
				'rate' => Woo_Bg_Dual_Currency_Core::get_conversion_rate(),
				'currency' => get_woocommerce_currency(),
				'symbol' => get_woocommerce_currency_symbol(),
				'is_bgn' => true
			));
		}
	}

	/**
	 * Display dual currency for product prices.
	 *
	 * @since    1.0.0
	 * @param    string     $price_html    The price HTML.
	 * @param    WC_Product $product       The product object.
	 * @return   string     The modified price HTML.
	 */
	public function display_dual_currency_price($price_html, $product)
	{
		// Prevent recursive calls and infinite loops
		static $processing = false;
		if ($processing) {
			return $price_html;
		}
		$processing = true;

		// Check if we should add dual currency
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency()) {
			$processing = false;
			return $price_html;
		}

		// Skip if price HTML is empty or already contains EUR symbol
		if (empty($price_html) || strpos($price_html, '€') !== false) {
			$processing = false;
			return $price_html;
		}

		// Handle sale prices differently
		if ($product->is_on_sale() && $product->get_sale_price() && $product->get_regular_price()) {
			$modified_price = $this->handle_sale_product_price($price_html, $product);
		} else {
			$modified_price = $this->handle_simple_product_price($price_html, $product);
		}

		$processing = false;
		return $modified_price;
	}

	/**
	 * Handle simple product price display.
	 *
	 * @since    1.0.0
	 * @param    string     $price_html    The price HTML.
	 * @param    WC_Product $product       The product object.
	 * @return   string     The modified price HTML.
	 */
	private function handle_simple_product_price($price_html, $product)
	{
		$price = $product->get_price();

		if (empty($price)) {
			return $price_html;
		}

		return Woo_Bg_Dual_Currency_Core::format_dual_currency($price_html, $price);
	}

	/**
	 * Handle sale product price display with dual currency.
	 *
	 * @since    1.0.0
	 * @param    string     $price_html    The price HTML.
	 * @param    WC_Product $product       The product object.
	 * @return   string     The modified price HTML.
	 */
	private function handle_sale_product_price($price_html, $product)
	{
		$regular_price = $product->get_regular_price();
		$sale_price = $product->get_sale_price();

		if (empty($regular_price) || empty($sale_price)) {
			return $price_html;
		}

		// Convert to EUR
		$regular_eur = Woo_Bg_Dual_Currency_Core::convert_bgn_to_eur($regular_price);
		$sale_eur = Woo_Bg_Dual_Currency_Core::convert_bgn_to_eur($sale_price);

		// Format dual currency prices
		$regular_dual = wc_price($regular_price) . ' / € ' . number_format($regular_eur, 2);
		$sale_dual = wc_price($sale_price) . ' / € ' . number_format($sale_eur, 2);

		// Use WooCommerce's sale price formatting but with dual currency
		return '<del aria-hidden="true">' . $regular_dual . '</del> <ins>' . $sale_dual . '</ins>';
	}

	/**
	 * Handle variable product price display.
	 *
	 * @since    1.0.0
	 * @param    string     $price_html    The price HTML.
	 * @param    WC_Product $product       The product object.
	 * @return   string     The modified price HTML.
	 */
	private function handle_variable_product_price($price_html, $product)
	{
		$min_price = $product->get_variation_price('min');
		$max_price = $product->get_variation_price('max');

		if ($min_price === $max_price) {
			return Woo_Bg_Dual_Currency_Core::format_dual_currency($price_html, $min_price);
		} else {
			return Woo_Bg_Dual_Currency_Core::format_price_range($min_price, $max_price);
		}
	}

	/**
	 * Handle grouped product price display.
	 *
	 * @since    1.0.0
	 * @param    string     $price_html    The price HTML.
	 * @param    WC_Product $product       The product object.
	 * @return   string     The modified price HTML.
	 */
	private function handle_grouped_product_price($price_html, $product)
	{
		$children = $product->get_children();

		if (empty($children)) {
			return $price_html;
		}

		$prices = array();
		foreach ($children as $child_id) {
			$child_product = wc_get_product($child_id);
			if ($child_product && $child_product->get_price()) {
				$prices[] = $child_product->get_price();
			}
		}

		if (empty($prices)) {
			return $price_html;
		}

		$min_price = min($prices);
		$max_price = max($prices);

		if ($min_price === $max_price) {
			return Woo_Bg_Dual_Currency_Core::format_dual_currency($price_html, $min_price);
		} else {
			return Woo_Bg_Dual_Currency_Core::format_price_range($min_price, $max_price);
		}
	}

	/**
	 * Display dual currency for cart item prices.
	 *
	 * @since    1.0.0
	 * @param    string $price_html    The price HTML.
	 * @param    array  $cart_item     Cart item data.
	 * @param    string $cart_item_key Cart item key.
	 * @return   string The modified price HTML.
	 */
	public function display_dual_currency_cart_price($price_html, $cart_item, $cart_item_key)
	{
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency()) {
			return $price_html;
		}

		$product = $cart_item['data'];
		$price = $product->get_price();

		return Woo_Bg_Dual_Currency_Core::format_dual_currency($price_html, $price);
	}

	/**
	 * Display dual currency for cart item subtotals.
	 *
	 * @since    1.0.0
	 * @param    string $subtotal_html The subtotal HTML.
	 * @param    array  $cart_item     Cart item data.
	 * @param    string $cart_item_key Cart item key.
	 * @return   string The modified subtotal HTML.
	 */
	public function display_dual_currency_cart_subtotal($subtotal_html, $cart_item, $cart_item_key)
	{
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency()) {
			return $subtotal_html;
		}

		$product = $cart_item['data'];
		$quantity = $cart_item['quantity'];
		$subtotal = $product->get_price() * $quantity;

		return Woo_Bg_Dual_Currency_Core::format_dual_currency($subtotal_html, $subtotal);
	}

	/**
	 * Display dual currency for variable product prices.
	 *
	 * @since    1.0.0
	 * @param    string              $price_html The price HTML.
	 * @param    WC_Product_Variable $product    Variable product object.
	 * @return   string              The modified price HTML.
	 */
	public function display_dual_currency_variable_price($price_html, $product)
	{
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency()) {
			return $price_html;
		}

		return $this->handle_variable_product_price($price_html, $product);
	}

	/**
	 * Display dual currency for grouped product prices.
	 *
	 * @since    1.0.0
	 * @param    string              $price_html The price HTML.
	 * @param    WC_Product_Grouped  $product    Grouped product object.
	 * @return   string              The modified price HTML.
	 */
	public function display_dual_currency_grouped_price($price_html, $product)
	{
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency()) {
			return $price_html;
		}

		return $this->handle_grouped_product_price($price_html, $product);
	}

	/**
	 * Display dual currency for shipping costs.
	 *
	 * @since    1.0.0
	 * @param    string $label   Shipping label.
	 * @param    object $method  Shipping method.
	 * @return   string The modified label.
	 */
	public function display_dual_currency_shipping($label, $method)
	{
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency()) {
			return $label;
		}

		// Extract cost from shipping method
		if (isset($method->cost) && $method->cost > 0) {
			$eur_cost = Woo_Bg_Dual_Currency_Core::convert_bgn_to_eur($method->cost);
			$cost_display = wc_price($method->cost) . ' / € ' . number_format($eur_cost, 2);

			// Replace the cost in the label
			$label = preg_replace(
				'/(' . preg_quote(wc_price($method->cost), '/') . ')/',
				$cost_display,
				$label
			);
		}

		return $label;
	}

	/**
	 * Display dual currency for cart subtotal.
	 *
	 * @since    1.0.0
	 * @param    string   $cart_subtotal The cart subtotal HTML.
	 * @param    bool     $compound      Whether the subtotal includes compound taxes.
	 * @param    WC_Cart  $cart          Cart object.
	 * @return   string   The modified subtotal HTML.
	 */
	public function display_dual_currency_cart_subtotal_total($cart_subtotal, $compound, $cart)
	{
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency()) {
			return $cart_subtotal;
		}

		$subtotal = $cart->get_displayed_subtotal();
		return Woo_Bg_Dual_Currency_Core::format_dual_currency($cart_subtotal, $subtotal);
	}

	/**
	 * Display dual currency for cart order total.
	 *
	 * @since    1.0.0
	 * @param    string $total_html The total HTML.
	 * @return   string The modified total HTML.
	 */
	public function display_dual_currency_cart_order_total($total_html)
	{
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency()) {
			return $total_html;
		}

		$total = WC()->cart->get_total('edit');
		return Woo_Bg_Dual_Currency_Core::format_dual_currency($total_html, $total);
	}

	/**
	 * Add EUR conversion to line item subtotal.
	 *
	 * @since    1.0.0
	 * @param    string   $subtotal Line subtotal.
	 * @param    array    $item     Order item.
	 * @param    WC_Order $order    Order object.
	 * @return   string   Modified subtotal with EUR conversion.
	 */
	public function add_eur_to_line_subtotal($subtotal, $item, $order)
	{
		if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency() || ! is_wc_endpoint_url('order-received')) {
			return $subtotal;
		}

		if (is_object($item) && method_exists($item, 'get_total')) {
			$bgn_total = $item->get_total();
			$eur_total = Woo_Bg_Dual_Currency_Core::convert_bgn_to_eur($bgn_total);
			$subtotal .= '<br><span style="font-size: 0.9em; color: #666; font-style: italic;">€ ' . number_format($eur_total, 2) . '</span>';
		}

		return $subtotal;
	}
}
