<?php

/**
 * Email functionality for dual currency display.
 *
 * @link       https://webtospec.com
 * @since      1.0.0
 *
 * @package    Woo_Bg_Dual_Currency
 * @subpackage Woo_Bg_Dual_Currency/includes
 */

/**
 * The email dual currency class.
 *
 * This class handles dual currency display in WooCommerce emails.
 *
 * @since      1.0.0
 * @package    Woo_Bg_Dual_Currency
 * @subpackage Woo_Bg_Dual_Currency/includes
 * @author     Web To Spec <grow@webtospec.com>
 */
class Woo_Bg_Dual_Currency_Email
{

    /**
     * Initialize email hooks.
     *
     * @since    1.0.0
     */
    public static function init()
    {
        // Only add hooks if WooCommerce is active and settings allow
        if (! class_exists('WooCommerce') || ! Woo_Bg_Dual_Currency_Core::show_in_emails()) {
            return;
        }

        // Hook into email price display
        add_filter('woocommerce_order_formatted_line_subtotal', array(__CLASS__, 'email_line_subtotal'), 9999, 3);
        add_filter('woocommerce_get_formatted_order_total', array(__CLASS__, 'email_order_total'), 9999, 2);

        // Hook into individual line items for better control
    }

    /**
     * Format dual currency in email line subtotal.
     *
     * @since    1.0.0
     * @param    string   $subtotal The subtotal HTML.
     * @param    object   $item     Order item.
     * @param    WC_Order $order    Order object.
     * @return   string   The modified subtotal HTML.
     */
    public static function email_line_subtotal($subtotal, $item, $order)
    {
        if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency()) {
            return $subtotal;
        }

        if (is_object($item) && method_exists($item, 'get_total')) {
            $line_total = $item->get_total();
            return self::format_email_price($subtotal, $line_total);
        }

        return $subtotal;
    }

    /**
     * Format dual currency in email order total.
     *
     * @since    1.0.0
     * @param    string   $formatted_total The formatted total.
     * @param    WC_Order $order           Order object.
     * @return   string   The modified total HTML.
     */
    public static function email_order_total($formatted_total, $order)
    {
        if (! Woo_Bg_Dual_Currency_Core::is_bgn_currency()) {
            return $formatted_total;
        }

        $total = $order->get_total();
        return self::format_email_price($formatted_total, $total);
    }

    /**
     * Format price for email display.
     *
     * @since    1.0.0
     * @param    string $price_html The original price HTML.
     * @param    float  $price      The price amount.
     * @return   string The formatted price HTML.
     */
    private static function format_email_price($price_html, $price)
    {
        if (! $price || $price <= 0) {
            return $price_html;
        }

        $eur_price = Woo_Bg_Dual_Currency_Core::convert_bgn_to_eur($price);

        // Create dual currency display for emails
        $bgn_formatted = wc_price($price, array('currency' => 'BGN'));
        $eur_part = '/ <span>€ ' . number_format($eur_price, 2) . '</span>';

        // If the original HTML contains price formatting, preserve it
        if (strpos($price_html, 'woocommerce-Price-amount') !== false) {
            return preg_replace(
                '/(<span class="woocommerce-Price-amount amount"[^>]*>.*?<\/span>)/',
                '$1 ' . $eur_part,
                $price_html
            );
        } else {
            // Simple text replacement
            $clean_price = wp_strip_all_tags($price_html);
            return $price_html . ' ' . $eur_part;
        }
    }
}
